// import { PrismaClient } from "@prisma/client";
// import bcrypt from 'bcrypt'
// import jwt from 'jsonwebtoken'
// import dotenv from 'dotenv'

const { PrismaClient } = require("@prisma/client");
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const dotenv = require('dotenv');




const prisma = new PrismaClient();

dotenv.config()


// Register a new user
module.exports.RegisterUser = async (req, res) => {
    const token = req.cookies.access_token;
    if (!token) {
        res.status(401).json({
            message: "unauthorized User"
        })
    } else {
        jwt.verify(token, process.env.JWT_SECRET, async (err, userInfo) => {
            if (err) {
                return res.status(401).json({
                    message: "unauthorized user"
                })
            } else {

                try {
                    // user info contains user id and email....check the role of the user
                    const role = await prisma.user.findUnique({
                        where: {
                            id: userInfo.id
                        },
                        include: {
                            role: true
                        }
                    })

                    if (role.role[0].role === "editor") {
                        res.status(400).json({
                            message: "only admins can create users"
                        })
                    } else if (role.role[0].role === "admin") {

                        const { firstname, lastname, username, email, password } = req.body;

                        const salt = await bcrypt.genSalt(10)
                        const hashedPassword = await bcrypt.hash(password, salt)

                        const UsedUser = await prisma.user.findUnique({
                            where: {
                                username
                            }
                        })
                        const UsedEmail = await prisma.user.findUnique({
                            where: {
                                email
                            }
                        })
                        if (UsedUser) {
                            return res.status(400).json({
                                message: "Username already taken"
                            })
                        } else if (UsedEmail) {
                            return res.status(400).json({
                                message: "email already exists"
                            })
                        } else {
                            try {
                                const user = await prisma.user.create({
                                    data: {
                                        firstname,
                                        lastname,
                                        username,
                                        email,
                                        password: hashedPassword,
                                        role: {
                                            connect: {
                                                id: 1
                                            }
                                        }

                                    }
                                })
                                res.status(201).json({
                                    message: "User created successfully",
                                })

                            } catch (err) {
                                console.log(err)
                                res.status(500).json({
                                    message: "Something went wrong"
                                })
                            }
                        }
                    }

                } catch (error) {
                    console.log("REGISTER_USER_ERROR", error)
                }
            }
        })
    }




}


// Login User
module.exports.UserLogin = async (req, res) => {
    const { username, email, password } = req.body;

    // Check if email exists in database
    const user = await prisma.user.findFirst({
        where: {
            OR: [
                { username: username },
                { email: email }
            ]
        }
    });
    if (!user) {
        res.status(401).json({
            message: "invalid email or password"
        })
    } else {
        // Check if password is correct
        const isPasswordCorrect = await bcrypt.compare(password, user.password)

        if (!user || !isPasswordCorrect) {

            res.status(401).json({
                message: "invalid email or password"
            })
        } else if (user && isPasswordCorrect) {
            // Assign Json web token
            try {

                const token = jwt.sign({ id: user.id, email: user.email, }, process.env.JWT_SECRET, { expiresIn: '1h' })
                const { password, email, ...activeUser } = user

                // Send token to client
                res.cookie('access_token', token, { httpOnly: true, maxAge: 1000 * 60 * 60 })
                res.status(200).json({
                    activeUser,
                    token
                })

            } catch (err) {
                console.log(err)
                res.status(500).json({ error: err.message })
            }

        }
    }
}

// Logout functionality

module.exports.Logout = async (req, res) => {
    res.cookie('access_token', '', { maxAge: 1 })
    res.status(200).json({
        message: "Logout Successful"
    })
}